/*____________________________________________________________________________
		Copyright (C) 2000 Network Associates, Inc.
        All rights reserved.

        $Id: CPath.cpp,v 1.12 2000/07/11 03:11:17 nryan Exp $
____________________________________________________________________________*/

#include "pgpClassesConfig.h"

#include <commdlg.h>
#include <dbt.h>
#include <shellapi.h>
#include <shlobj.h>

#include "CFile.h"

#include "CComLibrary.h"
#include "CComObject.h"
#include "CPath.h"
#include "CRegistryKey.h"
#include "UDynLink.h"
#include "UUnicode.h"
#include "UWinVersion.h"

_USING_PGP

_UNNAMED_BEGIN

// Constants

const HKEY	kRegistryMappedDriveRoot		= HKEY_CURRENT_USER;
const char	*kRegistryMappedDrive98Section	= "Network\\Persistent";
const char	*kRegistryMappedDriveNTSection	= "Network";

const char	*kInvalidPathChars		= "<>:\"|";
const char	*k83InvalidPathChars	= "<>:\"|,;*? ";
const char	*kShortCutExtension		= ".lnk";

_UNNAMED_END


// Class CPath member functions

CPath::CPath() : mFindHandle(INVALID_HANDLE_VALUE)
{
}

CPath::CPath(const char *str) : 
	CString(str), mFindHandle(INVALID_HANDLE_VALUE)
{
}

CPath::CPath(const CPath& path) : 
	CString(path.Get()), mFindHandle(INVALID_HANDLE_VALUE)
{
}

CPath& 
CPath::operator=(const char *str)
{
	CString::operator=(str);
	return *this;
}

CPath::~CPath()
{
	try
	{
		if (FindInProgress())
			FindClose();
	}
	catch (CComboError&) { }
}

CPath& 
CPath::operator=(const CPath& path)
{
	return operator=(path.Get());
}

PGPBoolean 
CPath::HasPlainLocalRoot() const
{
	return ((Length() >= 3) && 
		(UString::LetterToNumber(GetAt(0)) < kMaxDrives) && 
		(GetAt(1) == ':') && (IsSlashChar(GetAt(2))));
}

PGPBoolean 
CPath::IsPlainLocalRoot() const
{
	return ((Length() == 3) && HasPlainLocalRoot());
}

PGPBoolean 
CPath::IsNetworkedPath() const
{
	if (IsUNCPath())
		return TRUE;

	CPath	root;
	GetRootPart(root);

	return (GetDriveType(root) == DRIVE_REMOTE);
}

PGPBoolean 
CPath::IsShortCut() const
{
	if (Length() < 4)
		return FALSE;

	CPath	ext;
	GetExtensionPart(ext);

	return ext.CompareNoCase(kShortCutExtension);
}

PGPBoolean 
CPath::IsUNCPath() const
{
	if (!IsSlashChar(GetAt(0)) || !IsSlashChar(GetAt(1)))
		return FALSE;

	PGPUInt32	length		= Length();
	PGPUInt32	numSlashes	= 0;

	for (PGPUInt32	i = 0; i < length; i++)
	{
		if (IsSlashChar(GetAt(i)))
			numSlashes++;

		if (numSlashes >= 3)
			return TRUE;
	}

	return FALSE;
}

PGPBoolean 
CPath::EndsInExtension() const
{
	for (PGPInt32 i = Length(); i > 0; i--)
	{
		char	c	= GetAt(i);

		if (IsSlashChar(c))
			return FALSE;
		else if (c == '.')
			return TRUE;
	}

	return FALSE;
}

PGPBoolean 
CPath::IsLegalPath() const
{
	if (IsEmpty())
		return FALSE;

	PGPBoolean	supportsLong	= SupportsLongFilenames();
	PGPUInt32	i, startChar;

	if (HasPlainLocalRoot())
		startChar = 3;
	else if (IsUNCPath())
		startChar = 0;
	else
		return FALSE;

	PGPUInt32	lengthPath	= Length();

	if (supportsLong)
	{
		PGPUInt32	lengthInv	= strlen(kInvalidPathChars);

		for (i = startChar; i < lengthPath; i++)
		{
			for (PGPUInt32 j = 0; j < lengthInv; j++)
			{
				if (GetAt(i) == kInvalidPathChars[j])
					return FALSE;
			}
		}
	}
	else
	{
		PGPUInt32	lengthInv	= strlen(k83InvalidPathChars);

		for (i = startChar; i < lengthPath; i++)
		{
			for (PGPUInt32 j = 0; j < lengthInv; j++)
			{
				if (GetAt(i) == k83InvalidPathChars[j])
					return FALSE;
			}
		}
	}

	if (!supportsLong)
	{
		PGPInt32	chunkSize	= 0;

		for (i = lengthPath - 1; i >= 0; i--)
		{
			char	c	= GetAt(i);

			if (c == '.')
			{
				if (chunkSize > 3)
					return FALSE;

				chunkSize = 0;
			}

			if (IsSlashChar(c))
				break;

			chunkSize++;
		}

		if (chunkSize > 8)
			return FALSE;
	}

	return TRUE;
}

PGPBoolean 
CPath::IsValidDirectory() const
{
	PGPUInt32	attribs	= GetFileAttributes(Get());

	if (attribs == 0xFFFFFFFF)
		return FALSE;
	else
		return (attribs & FILE_ATTRIBUTE_DIRECTORY ? TRUE : FALSE);
}

PGPBoolean 
CPath::IsValidFile() const
{
	PGPUInt32	attribs	= GetFileAttributes(Get());

	if (attribs == 0xFFFFFFFF)
		return FALSE;
	else
		return (attribs & FILE_ATTRIBUTE_DIRECTORY ? FALSE : TRUE);
}

PGPBoolean 
CPath::IsCompressed() const
{
	return (GetFileAttributes(Get()) & FILE_ATTRIBUTE_COMPRESSED ? TRUE : 
		FALSE);
}

PGPBoolean 
CPath::IsDriveNetworkMapped() const
{
	pgpAssert(HasPlainLocalRoot());

	CString		keyName;
	keyName.Format("%c", GetAt(0));

	CString		regPathToMapped;

	if (UWinVersion::IsWin95Compatible())
		regPathToMapped = kRegistryMappedDrive98Section;
	else
		regPathToMapped = kRegistryMappedDriveNTSection;

	regPathToMapped.Append(kBackSlash);
	regPathToMapped.Append(keyName);

	// If key exists, drive is claimed.
	PGPBoolean	isMapped	 = FALSE;

	try
	{
		CRegistryKey	mappedDriveKey;
	
		mappedDriveKey.Open(kRegistryMappedDriveRoot, regPathToMapped, 
			KEY_READ);

		isMapped = TRUE;
	}
	catch (CComboError&) { }

	return isMapped;
}

PGPBoolean 
CPath::IsReadOnly() const
{
	PGPBoolean	isReadOnly;

	try
	{
		if (IsValidDirectory())
		{
			// Attempt to create a temporary file in the directory.
			CFile	theFile;
			CPath	tempName;

			if (!GetTempFileName(Get(), "pgp", 0, 
				tempName.GetBuffer(PFLConstants::kMaxPathLength)))
			{
				THROW_PGPERROR(kPGPError_FileOpFailed);
			}

			tempName.ReleaseBuffer();
			theFile.Delete(tempName);

			isReadOnly = FALSE;
		}
		else
		{
			// Try to write a byte to the file.
			CFile		theFile;
			PGPByte		firstByte;

			theFile.Open(Get());

			PGPUInt64	length	= theFile.GetLength();

			if (length > 0)
				theFile.Read(&firstByte, 0, 1);

			theFile.Write(&firstByte, 0, 1);

			if (length == 0)
				theFile.SetLength(0);

			isReadOnly = FALSE;
		}
	}
	catch (CComboError&)
	{
		isReadOnly = TRUE;
	}

	return isReadOnly;
}

PGPBoolean 
CPath::IsDirectoryEmpty() const
{
	CPath	allFiles(*this);

	allFiles.EndInSlash();
	allFiles.Append("*");

	CPath	curFile;

	if (!allFiles.FindFirstFile(curFile))
		THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());

	PGPBoolean	foundFiles	= FALSE;

	do
	{
		if ((curFile == ".") || (curFile == ".."))
			continue;

		foundFiles = TRUE;
	}
	while (allFiles.FindNextFile(curFile));

	allFiles.FindClose();

	return !foundFiles;
}

PGPUInt64 
CPath::GetFreeVolumeSpace() const
{
	// We must pass the root, not the path, to the system.
	CPath	root;
	GetRootPart(root);

	ULARGE_INTEGER	freeAvail;

	// There are two different functions - for OSR1 and for everything else.
	if (UWinVersion::IsWin95OSR2Compatible() || 
		UWinVersion::IsWinNT4Compatible())
	{
		PGPUInt32		result;
		ULARGE_INTEGER	totalBytes, totalFreeBytes;

		result = UDynLink::GetDiskFreeSpaceEx(root, &freeAvail, 
			&totalBytes, &totalFreeBytes);

		if (!result)
			THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());
	}
	else
	{
		unsigned long	spc, bps, freeClust, totalClust;

		if (!GetDiskFreeSpace(root, &spc, &bps, &freeClust, &totalClust))
			THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());

		freeAvail.QuadPart = bps * spc * freeClust;
	}

	return freeAvail.QuadPart;
}

PGPBoolean 
CPath::TestFreeVolumeSpace(PGPUInt64 bytes) const
{
	CPath	dir;
	GetDirPart(dir);

	// Open a temporary file in the directory.
	CPath	tempPath;

	try
	{
		GetTempFileName(dir, "pgp", 0, 
			tempPath.GetBuffer(PFLConstants::kMaxPathLength));
		tempPath.ReleaseBuffer();

		CFile	tempFile;
		tempFile.Open(tempPath, CFile::kCreateIfFlag);

		// Set the file to the desired length.
		tempFile.SetLength(bytes);

		// Make absolutely sure by closing and re-opening file.
		tempFile.Close();
		tempFile.Open(tempPath);

		if (tempFile.GetLength() != bytes)
			return FALSE;

		tempFile.Close();
		tempFile.Delete(tempPath);

		return TRUE;
	}
	catch (CComboError&)
	{
		return FALSE;
	}
}

PGPBoolean 
CPath::SupportsLongFilenames() const
{
	try
	{
		CPath	volRoot;
		GetVolumeRoot(volRoot);

		if (UWinVersion::IsWin2000Compatible())
		{
			CPath	realRoot;

			PGPBoolean	succeeded	= 
				UDynLink::Win2k_GetVolumeNameForVolumeMountPoint(volRoot, 
				realRoot.GetBuffer(PFLConstants::kMaxPathLength), 
				PFLConstants::kMaxPathLength);
			realRoot.ReleaseBuffer();

			if (succeeded)
			{
				volRoot = realRoot;
				volRoot.EndInSlash();
			}
		}

		DWORD	maxLength;

		if (!GetVolumeInformation(volRoot, NULL, 0, NULL, &maxLength, NULL, 
			NULL, 0))
		{
			THROW_ERRORS(kPGPError_VolumeOpFailed, GetLastError());
		}

		return (maxLength >= 255);
	}
	catch (CComboError&)
	{
		return TRUE;
	}
}

void 
CPath::GetVolumeFileSys(CString& fileSys) const
{
	CPath	volRoot;
	GetVolumeRoot(volRoot);

	if (UWinVersion::IsWin2000Compatible())
	{
		CPath	realRoot;

		PGPBoolean	succeeded	= 
			UDynLink::Win2k_GetVolumeNameForVolumeMountPoint(volRoot, 
			realRoot.GetBuffer(PFLConstants::kMaxPathLength), 
			PFLConstants::kMaxPathLength);
		realRoot.ReleaseBuffer();

		if (succeeded)
		{
			volRoot = realRoot;
			volRoot.EndInSlash();
		}
	}

	if (!GetVolumeInformation(volRoot, NULL, 0, NULL, NULL, NULL, 
		fileSys.GetBuffer(32), 32))
	{
		fileSys.ReleaseBuffer();
		THROW_ERRORS(kPGPError_VolumeOpFailed, GetLastError());
	}

	fileSys.ReleaseBuffer();
}

void 
CPath::GetVolumeRoot(CPath& volRoot) const
{
	if (UWinVersion::IsWin2000Compatible())
	{
		PGPBoolean	succeeded	= 
			UDynLink::Win2k_GetVolumePathName(Get(), 
			volRoot.GetBuffer(PFLConstants::kMaxPathLength), 
			PFLConstants::kMaxPathLength);
		volRoot.ReleaseBuffer();

		if (succeeded)
		{
			volRoot.EndInSlash();
			return;
		}
	}

	GetRootPart(volRoot);
}

void 
CPath::GetRootPart(CPath& root) const
{
	if (HasPlainLocalRoot())
	{
		Left(3, root);
	}
	else if (IsUNCPath())
	{
		PGPUInt32	length		= Length();
		PGPUInt32	numSlashes	= 0;

		for (PGPUInt32 i = 0; i < length; i++)
		{
			if (IsSlashChar(GetAt(i)))
				numSlashes++;

			root.Append(GetAt(i));

			if (numSlashes == 4)
				break;
		}
	}
}

void 
CPath::GetDirPart(CPath& dir) const
{
	CPath	fileName;
	GetFileNamePart(fileName);

	Left(Length() - fileName.Length(), dir);

	if (!dir.IsEmpty())
		dir.EndInSlash();
}

// GetDirPartSmart is like GetDirPart except it tacks on a working 
// directory path to incomplete directories.

void 
CPath::GetDirPartSmart(CPath& dir) const
{
	CPath	tempDir;

	if (!HasPlainLocalRoot() && !IsUNCPath())
	{
		tempDir.AssignCurrentDirectory();

		if (IsSlashChar(GetAt(0)))
			tempDir.DontEndInSlash();
		else
			tempDir.EndInSlash();

		tempDir.Append(Get());

		if (tempDir.IsValidDirectory())
			tempDir.EndInSlash();
	}
	else
	{
		tempDir = Get();
	}

	tempDir.GetDirPart(dir);
}

void 
CPath::GetExtensionPart(CPath& extension) const
{
	extension.Empty();
	PGPUInt32	length	= Length();

	for (PGPInt32 i = length; i > 0; i--)
	{
		char	c	= GetAt(i);

		if (IsSlashChar(c))
			break;

		extension.Prepend(c);

		if (c == '.')
			break;
	}
}

void 
CPath::GetFileNamePart(CPath& fileName) const
{
	fileName.Empty();
	PGPUInt32	length	= Length();

	for (PGPInt32 i = length; i > 0; i--)
	{
		char	c	= GetAt(i);

		if (IsSlashChar(c))
			break;

		fileName.Prepend(c);
	}
}

void 
CPath::GetServerPart(CPath& server) const
{
	pgpAssert(IsUNCPath());
	Mid(2, Find(kBackSlash, 3) - 1, server);
}

void 
CPath::GetSharePart(CPath& share) const
{
	pgpAssert(IsUNCPath());

	PGPUInt32	firstChar	= Find(kBackSlash, 3) + 1;
	PGPUInt32	lastChar	= Find(kBackSlash, 4);

	if (lastChar == -1)
		lastChar = Length();
	else
		lastChar--;

	Mid(firstChar, lastChar, share);
}

PGPUInt32 
CPath::GetFreeDrives(PGPBoolean ignoreAB)
{
	PGPUInt8	i;
	PGPUInt32	drives		= GetLogicalDrives();
	PGPUInt32	freeDrives	= 0;

	// Return array of drives we can actually use.	
	PGPUInt32	startDrive	= (ignoreAB ? 2 : 0);

	for (i = startDrive; i < kMaxDrives; i++)
	{
		if (!(drives & (1 << i)))
		{
			CPath	root;
			root.MakePlainLocalRoot(i);
			
			if (!root.IsDriveNetworkMapped())
				freeDrives |= (1 << i);
		}
	}

	return freeDrives;
}

void 
CPath::AddExtension(const char *ext)
{
	pgpAssertStrValid(ext);

	if (!IsEmpty() && !EndsInExtension())
		Append(ext);
}

void 
CPath::RemoveExtension()
{
	for (PGPInt32 i = Length(); i > 0; i--)
	{
		char	c	= GetAt(i);

		if (IsSlashChar(c))
		{
			break;
		}
		else if (c == '.')
		{
			Resize(i);
			break;
		}
	}
}

void 
CPath::GetDisplayFileName(CPath& displayName) const
{
	GetFileTitle(Get(), displayName.GetBuffer(PFLConstants::kMaxPathLength), 
		PFLConstants::kMaxPathLength);
	displayName.ReleaseBuffer();
}

void 
CPath::ConvertPathToLong()
{
	CPath	prefix;
	GetRootPart(prefix);

	CPath	postfix;

	Right(Length() - prefix.Length(), postfix);
	postfix.DontEndInSlash();

	Empty();

	while (!postfix.IsEmpty())
	{
		CPath	fullPath	= prefix;
		fullPath.Append(postfix);

		CPath	fileName;

		if (!fullPath.FindFirstFile(fileName))
			THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());

		fullPath.FindClose();

		int	lastSlash	= postfix.ReverseFind(kBackSlash);

		if (lastSlash == -1)
			postfix.Empty();
		else
			postfix.Resize(lastSlash);

		if (!IsEmpty())
			Prepend(kBackSlash);

		Prepend(fileName);
	}

	Prepend(prefix);

	if (IsValidDirectory())
		EndInSlash();
	else
		DontEndInSlash();
}

void 
CPath::ConvertPathToShort()
{
	CPath	temp;

	GetShortPathName(Get(), temp.GetBuffer(PFLConstants::kMaxPathLength), 
		PFLConstants::kMaxPathLength);

	Assign(temp);
}

void 
CPath::EndInSlash()
{
	if (!EndsInSlash())
		Append(kBackSlash);
}

void 
CPath::DontEndInSlash()
{
	if (EndsInSlash())
		Resize(Length() - 1);
}

void 
CPath::FixSlashes()
{
	for (PGPUInt16 i = 0; i < Length(); i++)
	{
		if (GetAt(i) == kForwardSlash)
			SetAt(i, kBackSlash);
	}
}

void 
CPath::ResolveLocalToUNC()
{
	using namespace UDynLink;

	if (UWinVersion::IsWin95Compatible())
	{
		CPath	tempPath	= Get();

		PGPBoolean	result	= Win95_WNetGetUniversalName(tempPath, 
			GetBuffer(PFLConstants::kMaxPathLength));
		ReleaseBuffer();

		if (!result)
			THROW_ERRORS(kPGPError_NetworkOpFailed, GetLastError());
	}
	else if (UWinVersion::IsWinNT4Compatible())
	{
		unsigned long	bufSize	= sizeof(UNIVERSAL_NAME_INFO) + 
			PFLConstants::kMaxPathLength * sizeof(WCHAR);

		CArray<PGPByte>	uniInfoBuf(bufSize);

		UNIVERSAL_NAME_INFO	*pUNI	= 
			reinterpret_cast<UNIVERSAL_NAME_INFO *>(uniInfoBuf.Get());

		PGPUInt32	result	= WinNT_WNetGetUniversalName(Get(), 
			UNIVERSAL_NAME_INFO_LEVEL, pUNI, &bufSize);

		if (result != NO_ERROR)
			THROW_ERRORS(kPGPError_NetworkOpFailed, result);

		Assign(pUNI->lpUniversalName);
	}
	else
	{
		pgpAssert(FALSE);
	}
}

void 
CPath::ResolveUNCToLocal()
{
	using namespace UDynLink;

	// extract strings we will use
	CPath	root;
	GetRootPart(root);

	CPath	notRoot;
	Right(Length() - root.Length(), notRoot);

	CPath	server;
	GetServerPart(server);

	CPath	share;
	GetSharePart(share);

	CPath	newRoot;

	// check equality of computer name manually, NetShareGetInfo is flaky
	CString	computerName;
	DWORD	maxCNSize	= MAX_COMPUTERNAME_LENGTH + 1;

	GetComputerName(computerName.GetBuffer(maxCNSize), &maxCNSize);

	if (!server.CompareNoCase(computerName))
		return;
		
	if (UWinVersion::IsWin95Compatible())
	{
		share.Uppercase();	// must be uppercase

		PGPUInt32	bufSize	= sizeof(share_info_50) + MAX_PATH + 
			MAXCOMMENTSZ + 2;

		CArray<PGPByte>	shareInfoBuf(bufSize);
		share_info_50	*pSI50	= reinterpret_cast<share_info_50 *>(
			shareInfoBuf.Get());

		unsigned short	cbTotalAvail;
		PGPUInt32		result	= Win95_NetShareGetInfo(NULL, share, 50, 
			reinterpret_cast<char *>(pSI50), bufSize, &cbTotalAvail);

		if (result != NERR_Success)
			THROW_ERRORS(kPGPError_NetworkOpFailed, result);

		newRoot = pSI50->shi50_path;
	}
	else if (UWinVersion::IsWinNT4Compatible())
	{

		CArray<WCHAR>	uniServer;
		UUnicode::AsciiToUni(server, uniServer);

		CArray<WCHAR>	uniSharePath;
		UUnicode::AsciiToUni(share, uniSharePath);

		SHARE_INFO_2	*pSI2;
		PGPUInt32		result	= WinNT_NetShareGetInfo(uniServer.Get(), 
			uniSharePath.Get(), 2, reinterpret_cast<LPBYTE *>(&pSI2));

		if (result != NERR_Success)
			THROW_ERRORS(kPGPError_NetworkOpFailed, result);

		UUnicode::UniToAscii(pSI2->shi2_path, newRoot);
		WinNT_NetApiBufferFree(pSI2);
	}

	Assign(newRoot);
	EndInSlash();
	Append(notRoot);
}

void 
CPath::ResolveLoopBackPath()
{
	try
	{
		if (IsNetworkedPath())
		{
			if (!IsUNCPath())
				ResolveLocalToUNC();

			ResolveUNCToLocal();
		}
	}
	catch (CComboError&) { }
}

void 
CPath::ResolveShortCut()
{
	pgpAssert(IsShortCut());

	PGPUInt32	result;
 
	// Load COM.
	CComLibrary	comLibrary;

	// Get a pointer to the IShellLink interface.
	CComObject<IShellLink>	pShellLink(CLSID_ShellLink, NULL, 
		CLSCTX_INPROC_SERVER, IID_IShellLink);

	// Get a pointer to the IPersistFile interface.
	CComObject<IPersistFile>	pIPF(static_cast<IPersistFile *>(
		pShellLink.QueryInterface(IID_IPersistFile)));

	// Ensure that the string is Unicode.
	CArray<WCHAR>	uniSCPath;
	UUnicode::AsciiToUni(Get(), uniSCPath);

	// Load the shortcut.
	result = pIPF->Load(uniSCPath.Get(), STGM_READ);

	if (result != S_OK)
		THROW_ERRORS(kPGPError_Win32COMOpFailed, result);

	// Resolve the link.
	result = pShellLink->Resolve(GetDesktopWindow(), SLR_ANY_MATCH);

	if (result != NOERROR)
		THROW_ERRORS(kPGPError_Win32COMOpFailed, result);

	// Get the path to the link target.
	WIN32_FIND_DATA	WFD;

	result = pShellLink->GetPath(GetBuffer(PFLConstants::kMaxPathLength), 
		PFLConstants::kMaxPathLength, &WFD, SLGP_SHORTPATH);
	ReleaseBuffer();

	if (result != NOERROR)
		THROW_ERRORS(kPGPError_Win32COMOpFailed, result);
}

void 
CPath::Canonicalize(const char *extension)
{
	FixSlashes();

	if (IsShortCut())
		ResolveShortCut();

	if (IsValidDirectory())
		EndInSlash();

	// If looped back, resolve it a to local path. This check WILL fail
	// under non-Admin accounts under NT, we take care of that later.

	ResolveLoopBackPath();

	// Fill out filename if this is a file.
	if (!IsValidDirectory())
	{
		CPath	dir;
		GetDirPartSmart(dir);

		CPath	bareName;
		GetFileNamePart(bareName);

		char *pFilePart;

		if (!::SearchPath(dir, bareName, extension, 
			PFLConstants::kMaxPathLength, 
			GetBuffer(PFLConstants::kMaxPathLength), &pFilePart))
		{
			ReleaseBuffer();
			THROW_PGPERROR(kPGPError_FileNotFound);
		}

		ReleaseBuffer();
	}

	// Convert to long pathname form.
	ConvertPathToLong();
}

PGPBoolean 
CPath::FindFirstFile(CPath& fileName)
{
	pgpAssert(!FindInProgress());

	WIN32_FIND_DATA	findData;
	mFindHandle	= ::FindFirstFile(Get(), &findData);

	if (mFindHandle == INVALID_HANDLE_VALUE)
		return FALSE;

	fileName = findData.cFileName;
	return TRUE;
}

PGPBoolean 
CPath::FindNextFile(CPath& fileName)
{
	pgpAssert(FindInProgress());

	WIN32_FIND_DATA	findData;

	if (!::FindNextFile(mFindHandle, &findData))
		return FALSE;

	fileName = findData.cFileName;
	return TRUE;
}

void 
CPath::FindClose()
{
	pgpAssert(FindInProgress());

	::FindClose(mFindHandle);
	mFindHandle = INVALID_HANDLE_VALUE;
}

void 
CPath::AssignCurrentDirectory()
{
	PGPUInt32	result	= GetCurrentDirectory(PFLConstants::kMaxPathLength, 
		GetBuffer(PFLConstants::kMaxPathLength));
	ReleaseBuffer();

	if (result == 0)
		THROW_ERRORS(kPGPError_FileOpFailed, result);
}

void 
CPath::BroadcastDriveMessage(WPARAM msg) const
{
	pgpAssert(HasPlainLocalRoot());

	DEV_BROADCAST_VOLUME	DBV;
	PGPInt32				result;

	DBV.dbcv_size		= sizeof(DBV); 
	DBV.dbcv_devicetype = DBT_DEVTYP_VOLUME; 
	DBV.dbcv_reserved	= 0;
	DBV.dbcv_unitmask	= 1 << GetDriveNum();
	DBV.dbcv_flags		= DBTF_MEDIA;

	result = UDynLink::BroadcastSystemMessage(
		BSF_IGNORECURRENTTASK | BSF_POSTMESSAGE, NULL, WM_DEVICECHANGE, msg, 
		reinterpret_cast<LPARAM>(&DBV));

	if (result < 1)
		THROW_PGPERROR(kPGPError_Win32WindowOpFailed);
}

void 
CPath::MakePlainLocalRoot(PGPUInt8 drive)
{
	Format("%c:\\", UString::NumberToLetter(drive));
}

void 
CPath::RecurseCreateDir()
{
	CPath	prefix;
	GetRootPart(prefix);

	CPath	postfix;
	Right(Length() - prefix.Length(), postfix);

	postfix.EndInSlash();

	CPath	fullDir	= prefix;

	while (!postfix.IsEmpty())
	{
		PGPInt32	nextSlash	= postfix.Find(kBackSlash);

		CPath	curDir;
		postfix.Left(nextSlash + 1, curDir);

		fullDir += curDir;

		if (!fullDir.IsValidDirectory() && !CreateDirectory(fullDir, NULL))
			THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());

		CPath	temp;

		postfix.Right(postfix.Length() - nextSlash - 1, temp);
		postfix = temp;
	}
}

void 
CPath::Browse() const
{
	SHELLEXECUTEINFO	sei;
	pgpClearMemory(&sei, sizeof(sei));

	sei.cbSize = sizeof(sei);
	sei.fMask = SEE_MASK_FLAG_NO_UI;
	sei.lpVerb = "explore";
	sei.lpFile = Get();
	sei.nShow = SW_SHOWNORMAL;

	ShellExecuteEx(&sei);
}

PGPBoolean 
CPath::AreShowingExtensions()
{
	CPath	appPath;

	// Get the pathname of the application.
	PGPUInt32	result	= GetModuleFileName(NULL, appPath.GetBuffer(
		PFLConstants::kMaxPathLength), PFLConstants::kMaxPathLength);
	appPath.ReleaseBuffer();

	// Get the display name.
	CPath	displayName;
	appPath.GetDisplayFileName(displayName);

	// Return whether it has an extension.
	return displayName.EndsInExtension();
}
